package gov.cms.grouper.snf.component.v100.logic;

import gov.cms.grouper.snf.SnfTables;
import gov.cms.grouper.snf.model.Assessment;
import gov.cms.grouper.snf.model.ClinicalCategory;
import gov.cms.grouper.snf.model.CognitiveLevel;
import gov.cms.grouper.snf.model.SnfDiagnosisCode;
import gov.cms.grouper.snf.model.table.ClinicalCategoryMappingRow;
import gov.cms.grouper.snf.model.table.SlpCmgRow;
import gov.cms.grouper.snf.util.ClaimInfo;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Test;

public class SpeechLogicTest {

  public static final int version = 100;

  @Test
  public void testStep2() {
    String pdxClinicalCategory = "Acute Infections";
    try {
      SpeechLogic.step2SlpClinicalCategory(-1, pdxClinicalCategory);
      Assertions.fail();
    } catch (Exception ex) {
      Assertions.assertTrue(true);
    }

    String expected = "Non-Neurologic";
    String actual =
        SpeechLogic.step2SlpClinicalCategory(SpeechLogicTest.version, pdxClinicalCategory);
    Assertions.assertEquals(expected, actual);

    pdxClinicalCategory = "Acute Neurologic";
    expected = "Acute Neurologic";
    actual = SpeechLogic.step2SlpClinicalCategory(SpeechLogicTest.version, pdxClinicalCategory);
    Assertions.assertEquals(expected, actual);

    pdxClinicalCategory = "Pulmonary";
    expected = "Non-Neurologic";
    actual = SpeechLogic.step2SlpClinicalCategory(SpeechLogicTest.version, pdxClinicalCategory);
    Assertions.assertEquals(expected, actual);

  }

  @Test
  public void testStep3() {
    boolean expected = true;
    boolean actual = SpeechLogic.step3HasSlpRelatedComorbidities(SpeechLogicTest.version,
        () -> true, Collections.emptyList());
    Assertions.assertEquals(expected, actual);

    expected = false;
    actual = SpeechLogic.step3HasSlpRelatedComorbidities(SpeechLogicTest.version, () -> false,
        Collections.emptyList());
    Assertions.assertEquals(expected, actual);

    List<SnfDiagnosisCode> secondaryDiagnoses =
        Collections.singletonList(new SnfDiagnosisCode("I69090", null, null, null));
    expected = true;
    actual = SpeechLogic.step3HasSlpRelatedComorbidities(SpeechLogicTest.version, () -> false,
        secondaryDiagnoses);
    Assertions.assertEquals(expected, actual);

    secondaryDiagnoses =
        Collections.singletonList(new SnfDiagnosisCode("S93301S", null, null, null));
    expected = false;
    actual = SpeechLogic.step3HasSlpRelatedComorbidities(SpeechLogicTest.version, () -> false,
        secondaryDiagnoses);
    Assertions.assertEquals(expected, actual);

    secondaryDiagnoses = Collections.singletonList(new SnfDiagnosisCode("A00", null, null, null));
    expected = false;
    actual = SpeechLogic.step3HasSlpRelatedComorbidities(SpeechLogicTest.version, () -> false,
        secondaryDiagnoses);
    Assertions.assertEquals(expected, actual);

  }

  @Test
  public void testStep4() {
    // CognitiveLevel
    boolean expected = false;
    boolean actual = SpeechLogic.step4HasCognitiveImpairment(CognitiveLevel.COGNITIVELY_INTACT);
    Assertions.assertEquals(expected, actual);

    expected = true;
    actual = SpeechLogic.step4HasCognitiveImpairment(CognitiveLevel.MILDLY_IMPAIRED);
    Assertions.assertEquals(expected, actual);

    expected = true;
    actual = SpeechLogic.step4HasCognitiveImpairment(CognitiveLevel.MODERATELY_IMPAIRED);
    Assertions.assertEquals(expected, actual);

    expected = true;
    actual = SpeechLogic.step4HasCognitiveImpairment(CognitiveLevel.SEVERELY_IMPARED);
    Assertions.assertEquals(expected, actual);

  }

  @Test
  public void testStep5() {
    int expected = 3;
    int actual = SpeechLogic.step5NumberOfConditionsPresent(
        () -> ClinicalCategoryMappingRow.SLP_Acute_Neurologic, () -> true, () -> true);
    Assertions.assertEquals(expected, actual);

    expected = 2;
    actual = SpeechLogic.step5NumberOfConditionsPresent(() -> null, () -> true, () -> true);
    Assertions.assertEquals(expected, actual);

    expected = 1;
    actual = SpeechLogic.step5NumberOfConditionsPresent(() -> null, () -> false, () -> true);
    Assertions.assertEquals(expected, actual);

    expected = 0;
    actual = SpeechLogic.step5NumberOfConditionsPresent(() -> null, () -> false, () -> false);
    Assertions.assertEquals(expected, actual);

  }


  @Test
  public void testStep6To8() {
    int expected = SpeechLogic.EITHER;
    int actual = SpeechLogic
        .step6To8PresenceOfMechanicallyAlteredDietOrSwallowingDisorder(() -> true, () -> false);
    Assertions.assertEquals(expected, actual);

    expected = SpeechLogic.BOTH;
    actual = SpeechLogic.step6To8PresenceOfMechanicallyAlteredDietOrSwallowingDisorder(() -> true,
        () -> true);
    Assertions.assertEquals(expected, actual);

    expected = SpeechLogic.NEITHER;
    actual = SpeechLogic.step6To8PresenceOfMechanicallyAlteredDietOrSwallowingDisorder(() -> false,
        () -> false);
    Assertions.assertEquals(expected, actual);

    expected = SpeechLogic.EITHER;
    actual = SpeechLogic.step6To8PresenceOfMechanicallyAlteredDietOrSwallowingDisorder(() -> false,
        () -> true);
    Assertions.assertEquals(expected, actual);

  }

  @Test
  public void testStep9() {
    List<SlpCmgRow> expecteds = new ArrayList<>(SnfTables.slpCmgTable.values().stream()
        .flatMap(listContainer -> listContainer.stream()).collect(Collectors.toList()));

    expecteds.sort((o1, o2) -> {
      Integer p1 = o1.getPresenceOfAncSlCi();
      Integer p2 = o2.getPresenceOfAncSlCi();
      Integer m1 = o1.getMechAltDietOrSwallowDisorder();
      Integer m2 = o2.getMechAltDietOrSwallowDisorder();
      int result = p1.compareTo(p2);
      if (result == 0) {
        result = m1.compareTo(m2);
      }
      return result;

    });

    int index = 0;
    for (int i = 0; i <= 3; i++) {
      final int conditionsPresentCount = i;
      for (int j = 0; j <= 2; j++) {
        final int mechAltDietOrSwallowDisorder = j;
        SlpCmgRow actual = SpeechLogic.step9PdpmSlpClassification(SpeechLogicTest.version,
            () -> conditionsPresentCount, () -> mechAltDietOrSwallowDisorder);
        SlpCmgRow expected = expecteds.get(index++);
        Assertions.assertEquals(expected, actual);
      }
    }
  }

  @Test
  public void testExec() {
    List<Assessment> assessments = Collections.emptyList();
    ClaimInfo claim = ClaimInfo.of(100, false, assessments);
    List<SnfDiagnosisCode> secondaryDiagnosisCodes =
        Collections.singletonList(new SnfDiagnosisCode("C320", null, null, null));
    SpeechLogic com =
        new SpeechLogic(100, claim, secondaryDiagnosisCodes, CognitiveLevel.MILDLY_IMPAIRED,
            ClinicalCategory.NON_ORTHOPEDIC_SURGERY, () -> true, () -> false);

    Assertions.assertEquals("SH", com.exec());
  }

}
